#  
#  WordCompounder --- Executable file name: GoMusubi
#  Copyright(C) 2021 Kaoru Sagara and Syugo Nakamura 
#  This software is released under any of the GPL (see the file GPL), the LGPL(see the file LGPL), or the BSD License (see the file BSD).

import MeCab
import os
import unicodedata
import traceback
from components.exceptions import MecabTaggerValueError, LoadFailureException, SaveFailureException


class MecabParser:
  def __init__(self, sys_dic_dirpath, user_dic_filepath):
    """
    コンストラクタ。

    Args:
        sys_dic_dirpath (str): システム辞書のフォルダパス
        user_dic_filepath (str): ユーザ辞書のファイルパス
    """
    if sys_dic_dirpath:
      sys_dic_dirpath = os.path.abspath(sys_dic_dirpath)
    if user_dic_filepath:
      user_dic_filepath = os.path.abspath(user_dic_filepath)
    
    self._sys_dic_dirpath = sys_dic_dirpath
    self._user_dic_filepath = user_dic_filepath
    self._tagger = self._create_tagger()
    self._result_list = []
    self._input_text = ""

  # ==============================================================
  # 公開関数
  # ==============================================================
  def parse(self, text, normalize=True):
    """
    MeCabでの形態素解析を実行します。
    結果はself.result_listに格納されます。

    Args:
        text (str): 入力テキスト
        normalize (bool, optional): NFKC正規化を有効にする場合はTrue. Defaults to True.
    """
    # サロゲート文字対策
    text = text.encode('utf-16', 'surrogatepass').decode('utf-16', errors='ignore')

    # 正規化した結果を格納
    if normalize:
      text = unicodedata.normalize('NFKC', text)
      text = text.replace(' ', '　')  # MeCabでパースすると半角スペースが消滅するので全角に変える

    parse_result_list = []
    line_list = text.splitlines()

    # 1行ずつ形態素解析する。
    for line_cnt, line in enumerate(line_list):
      if line_cnt > 0:
        # 改行コードも記録する。
        parse_result_list.append(['', ['改行',]])
      
      node = self._tagger.parseToNode(line)
      
      while node:
        surface = node.surface.rstrip()
        feature_list = node.feature.split(',')
        if 'BOS/EOS' not in feature_list:
          parse_result_list.append([surface, feature_list])
        node = node.next
    
    # フィルタ関数を通して結果を確定させる。
    self._result_list = self._filter(parse_result_list)

    self._make_input_text()
    return self._result_list

  def load(self, filepath, encoding='utf-8'):
    """
    形態素解析の結果を読み込む。

    Args:
        filepath (str): TSV形式の形態素解析結果ファイル
        encoding (str, optional): エンコーディング形式. Defaults to 'utf-8'.
    Raises:
      LoadFailureException: ファイル読み込みに失敗したときの例外
      """
    with open(filepath, mode='r', encoding=encoding, errors='ignore') as f:
      lines = f.readlines()

    parse_result_list = []
    for line in lines:
      try:
        splitted_line = line.split('\t')
        if len(splitted_line) >= 2:
          surface = splitted_line[0]
          features_str = splitted_line[1]
      except ValueError:
        raise LoadFailureException("形態素解析結果ファイルのフォーマットが不正です。")
      features = features_str.rstrip('\n').split(',')
      parse_result_list.append([surface, features])

    self._result_list = parse_result_list
    self._make_input_text()

  def save(self, filepath, fmt='default', encoding='utf-8'):
    """
    形態素解析の結果をテキスト保存する。

    Args:
        filepath (str): 保存先のファイルパス
        fmt (str, optional): 出力フォーマット['wakati', 'default']. Defaults to 'wakati'.
        encoding (str, optional): エンコーディング形式. Defaults to 'utf-8'.
    """
    output_text = self.make_output(self.result_list, fmt)

    try:
      with open(filepath, mode='w', encoding=encoding, errors='ignore') as f:
        f.write(output_text)
    except PermissionError:
      raise SaveFailureException("ファイルに書き込みできません。")

  # ==============================================================
  # プロパティ
  # ==============================================================
  @property
  def result_list(self):
    """
    パース結果を格納したリスト。
    各形態素が[surface, [feature1, feature2,...]]のフォーマットで格納されています。

    Returns:
        list: パース結果を格納したリスト
    """
    return self._result_list

  @property
  def input_text(self):
    """
    入力テキストを返す。
    """
    return self._input_text

  # ==============================================================
  # 内部関数
  # ==============================================================
  def _make_input_text(self):
    """
    パース結果から入力テキストを復元する。
    Returns:
        str: 入力テキスト
    """
    self._input_text = ""
    for parse_result in self.result_list:
      surface, features = parse_result
      if '改行' in features:
        self._input_text += '\n'
      elif '空白' in features:
        self._input_text += '　'
      else:
        self._input_text += surface
    return self._input_text

  def _filter(self, result_list):
    """
    Mecabによるパース結果を変更する処理。

    Args:
        result_list (list): Mecabのパース結果が格納されたリスト

    Returns:
        list: 新しいパース結果のリスト
    """
    new_result_list = []
    for parse_result in result_list:
      surface, feature_list = parse_result
      # 特殊記号を変換
      if surface == '"':
        surface = '”'
      elif surface == '\'':
        surface = '’'
      new_result_list.append([surface, feature_list])
    return new_result_list
  
  def _make_arg(self):
    """
    Mecabに与えるコマンド引数を生成します。

    Returns:
        str: 生成されたコマンド引数
    """
    mecab_arg = ""

    if self.sys_dic_dirpath:
      mecab_arg += r"-d '{sys_dic}'".format(
        sys_dic=self.sys_dic_dirpath,
      )
    
    if self.user_dic_filepath:
      mecab_arg += r" -u '{user_dic}'".format(
        user_dic=self.user_dic_filepath
      )

    return mecab_arg

  def _create_tagger(self):
    """
    MecabのTagerを生成します。
    """
    mecab_arg = self._make_arg()
    try:
      tagger = MeCab.Tagger(mecab_arg)
    except Exception:
      raise MecabTaggerValueError("辞書ファイルが正しくありません。")
    return tagger

  @staticmethod
  def make_output(result_list, fmt='default'):
    """
    形態素解析の結果を出力します。

    Args:
        result_list (list): Mecabのパース結果が格納されたリスト
        fmt (str, optional): 出力フォーマット['default', 'wakati']. Defaults to 'wakati'.

    Returns:
        str: 分かち書きされた文字列
    """
    output = ""

    if fmt == 'wakati':
      surface_list = [result[0] for result in result_list]
      output = " ".join(surface_list)
    else:
      line_list = ["{}\t{}".format(result[0], ",".join(result[1])) for result in result_list]
      output = "\n".join(line_list)

    return output

  @property
  def sys_dic_dirpath(self):
    """
    システム辞書のフォルダパスを返す。

    Returns:
        str: システム辞書のフォルダパス
    """
    return self._sys_dic_dirpath

  @property
  def user_dic_filepath(self):
    """
    ユーザ辞書のファイルパスを返す。

    Returns:
        str: ユーザ辞書のファイルパス
    """
    return self._user_dic_filepath
